package com.dayouzc.e2e.core.util;

import android.annotation.SuppressLint;
import android.util.Log;

import com.dayouzc.e2e.core.DYUtils;
import com.dayouzc.e2e.core.constant.common.BaseConstant;
import com.orhanobut.logger.AndroidLogAdapter;
import com.orhanobut.logger.DiskLogAdapter;
import com.orhanobut.logger.FormatStrategy;
import com.orhanobut.logger.Logger;
import com.orhanobut.logger.PrettyFormatStrategy;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * Android开发调试日志工具类[支持保存到SD卡
 * 简单日志输出才有框架模式、其他日志输出需要传入Tag 才有Android原生模式
 * @author 张金盼
 * @date 2018/5/23
 */
@SuppressLint("SimpleDateFormat")
public class LogUtils {

    public static final String CACHE_DIR_NAME = "DayouLog";
    /**
     * 是否输出日志
     */
    public static boolean isDebugModel = false;
    /**
     * 是否保存调试日志
     */
    public static boolean isSaveDebugInfo = false;
    /**
     * 是否保存报错日志
     */
    public static boolean isSaveCrashInfo = false;

    /**
     * isDebugModel  是否输出日志  默认 false
     * isSaveDebugInfo 是否保存调试日志 默认 false
     * isSaveCrashInfo 是否保存报错日志 默认 false
     */
    public static void init(boolean isDebugModel, boolean isSaveDebugInfo, boolean isSaveCrashInfo) {
        FormatStrategy formatStrategy = PrettyFormatStrategy.newBuilder()
                .showThreadInfo(true)  // (可选) 显示线程信息（默认true）
//                .methodCount(0)         // (可选) 有多少方法线要显示。默认值2
//                .methodOffset(3)        // (可选) 跳过堆栈跟踪中调用的一些方法。默认的5
//        .logStrategy(customLog) // (可选) 将日志策略更改为打印输出。默认LogCat
                .tag(BaseConstant.TAG)   // (可选) 为每个日志定制标记。默认PRETTY_LOGGER
                .build();
        Logger.addLogAdapter(new AndroidLogAdapter(formatStrategy));
        LogUtils.isDebugModel = isDebugModel;
        LogUtils.isSaveDebugInfo = isSaveDebugInfo;
        LogUtils.isSaveCrashInfo = isSaveCrashInfo;

        if (isSaveCrashInfo){
            Logger.addLogAdapter(new DiskLogAdapter());
        }
    }


    public static void v(final String tag, final String msg) {
        if (isDebugModel) {
            Log.v(tag, "--> " + msg);
//            Logger.t(tag).v(msg);
        }
    }

    public static void d(final String tag, final String msg) {
        if (isDebugModel) {
            Log.d(tag, "--> " + msg);
//            Logger.t(tag).d(msg);
        }
    }

    public static void i(final String tag, final String msg) {
        if (isDebugModel) {
            Log.i(tag, "--> " + msg);
//            Logger.t(tag).i(msg);
        }
    }

    public static void w(final String tag, final String msg) {
        if (isDebugModel) {
            Log.w(tag, "--> " + msg);
//            Logger.t(tag).v(msg);
        }
    }

    /**
     * 调试日志，便于开发跟踪。
     *
     * @param tag
     * @param msg
     */
    public static void e(final String tag, final String msg) {
        if (isDebugModel) {
            Log.e(tag, "--> " + msg);
//            Logger.t(tag).e(msg);
        }

        if (isSaveDebugInfo) {
            new Thread() {
                @Override
                public void run() {
                    write(time() + tag + " --> " + msg + "\n");
                }
            }.start();
        }
    }

    public static void v(final String msg) {
        Logger.v(msg);
    }

    public static void d(final String msg) {
        Logger.d(msg);
    }

    public static void i(final String msg) {
        Logger.d(msg);
    }

    public static void w(final String msg) {
        Logger.w(msg);
    }

    public static void e(final String msg) {
        Logger.e(msg);
        if (isSaveDebugInfo) {
            new Thread() {
                @Override
                public void run() {
                    write(time() + " --> " + msg + "\n");
                }
            }.start();
        }
    }
    /**
     * try catch 时使用，上线产品可上传反馈。
     *
     * @param tag
     * @param tr
     */
    public static void e(final String tag, final Throwable tr) {
        if (isSaveCrashInfo) {
            new Thread() {
                @Override
                public void run() {
                    write(time() + tag + " [CRASH] --> "
                            + getStackTraceString(tr) + "\n");
                }
            }.start();
        }
    }

    /**
     * 获取捕捉到的异常的字符串
     *
     * @param tr
     * @return
     */
    public static String getStackTraceString(Throwable tr) {
        if (tr == null) {
            return "";
        }

        Throwable t = tr;
        while (t != null) {
            if (t instanceof UnknownHostException) {
                return "";
            }
            t = t.getCause();
        }

        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        tr.printStackTrace(pw);
        return sw.toString();
    }

    /**
     * 标识每条日志产生的时间
     *
     * @return
     */
    private static String time() {
        return "[" + new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date(
                System.currentTimeMillis())) + "] ";
    }

    /**
     * 以年月日作为日志文件名称
     *
     * @return
     */
    private static String date() {
        return new SimpleDateFormat("yyyy-MM-dd").format(new Date(System
                .currentTimeMillis()));
    }

    /**
     * 保存到日志文件
     *
     * @param content
     */
    public static synchronized void write(String content) {
        try {
            FileWriter writer = new FileWriter(getFile(), true);
            writer.write(content);
            writer.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 获取日志文件路径
     *
     * @return
     */
    public static String getFile() {
        File sdDir = null;

        sdDir = DYUtils.getContext().getExternalCacheDir();

        if (null == sdDir){
            sdDir = DYUtils.getContext().getExternalCacheDir();
        }

        File cacheDir = new File(sdDir + File.separator + CACHE_DIR_NAME);
        if (!cacheDir.exists()) {
            cacheDir.mkdir();
        }

        File filePath = new File(cacheDir + File.separator + date() + ".txt");
        if (!filePath.exists()) {
            try {
                //在指定的文件夹中创建文件
                filePath.createNewFile();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        LogUtils.d("日志保存目录为："+sdDir +"详细地址为："+filePath);

        return filePath.toString();
    }

}