package com.dayouzc.e2e.core.util.system;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.os.Environment;
import android.os.PowerManager;
import android.provider.Settings;
import android.support.annotation.RequiresPermission;
import android.telephony.TelephonyManager;
import android.webkit.WebSettings;
import android.webkit.WebView;

import com.dayouzc.e2e.core.DYUtils;
import com.dayouzc.e2e.core.bean.DeviceData;
import com.dayouzc.e2e.core.bean.DeviceDataDTO;
import com.dayouzc.e2e.core.consts.UIConsts;
import com.dayouzc.e2e.core.util.DevicesUtils;
import com.dayouzc.e2e.core.util.LoginUtils;
import com.dayouzc.e2e.core.util.StringUtils;
import com.dayouzc.e2e.core.util.app.AppUtils;
import com.dayouzc.e2e.core.util.common.ShellUtils;
import com.dayouzc.e2e.core.util.data.JsonUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Field;
import java.net.NetworkInterface;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;

import static android.Manifest.permission.ACCESS_WIFI_STATE;
import static android.Manifest.permission.INTERNET;
import static com.dayouzc.e2e.core.util.DevicesUtils.getYPSN;

/**
 * TODO 设备相关工具类
 *
 * @author 何鑫
 * @date 2020/3/17 下午 05:18
 */
public final class DeviceUtils {
    private final static String ANDROID = "Android";
    @SuppressLint("SdCardPath")
    private static String YOUPIE_PATH = "/mnt/sdcard/dycsctek/token/youpie.txt";
    private DeviceUtils() {
        throw new UnsupportedOperationException("u can't instantiate me...");
    }

    /**
     * 获取设备的所有信息
     *
     * @return
     */
    public static Map<String, String> getDeviceInfos() {
        Map<String, String> deviceInfos = new TreeMap<>();
        // 反射机制
        Field[] fields = Build.class.getDeclaredFields();
        // 迭代Build的字段key-value 此处的信息主要是为了在服务器端手机各种版本手机报错的原因
        try {
            // 获得包管理器
            PackageManager packageManager = AppUtils.getPackageManager();
            // 得到该应用的信息，即主Activity
            PackageInfo packageInfo = packageManager.getPackageInfo(AppUtils.getPackageName(), PackageManager.GET_ACTIVITIES);
            if (packageInfo != null) {
                deviceInfos.put("packageName", packageInfo.packageName);
                deviceInfos.put("versionName", packageInfo.versionName);
                deviceInfos.put("versionCode", String.valueOf(packageInfo.versionCode));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        for (Field field : fields) {
            try {
                field.setAccessible(true);
                deviceInfos.put(field.getName(), StringUtils.toString(field.get("")));
            } catch (IllegalArgumentException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        return deviceInfos;
    }

    /**
     * 获取设备的所有信息
     *
     * @return
     */
    public static String getDisplayDeviceInfos() {
        Map<String, String> deviceInfos = getDeviceInfos();
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, String> entry : deviceInfos.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            sb.append(key + "=" + value + "\r\n");
        }
        return sb.toString();
    }

    /**
     * 获取安卓系统版本
     *
     * @return
     */
    public static String getAndroidVersionName() {
        return ANDROID + " " + getSDKVersionName() + ", API " + getSDKVersionCode();
    }

    /**
     * 获取设备系统版本号
     *
     * @return 设备系统版本号
     */
    public static String getSDKVersionName() {
        return Build.VERSION.RELEASE;
    }

    /**
     * 获取设备系统版本码
     *
     * @return 设备系统版本码
     */
    public static int getSDKVersionCode() {
        return Build.VERSION.SDK_INT;
    }

    /**
     * 获取设备品牌
     *
     * @return
     */
    public static String getDeviceBrand() {
        return Build.BOARD + "  " + Build.MANUFACTURER;
    }

    /**
     * 获取设备 AndroidID
     *
     * @return AndroidID
     */
    @SuppressLint("HardwareIds")
    public static String getAndroidID() {
        return Settings.Secure.getString(
                DYUtils.getContext().getContentResolver(),
                Settings.Secure.ANDROID_ID
        );
    }

    /**
     * 获取设备厂商
     * <p>如 Xiaomi</p>
     *
     * @return 设备厂商
     */

    public static String getManufacturer() {
        return Build.MANUFACTURER;
    }

    /**
     * 获取硬件信息
     *
     * @return
     */
    public static String getHardware() {
        return Build.HARDWARE;
    }

    /**
     * 获取产品信息
     *
     * @return
     */
    public static String getProduct() {
        return Build.PRODUCT;
    }

    /**
     * 获取设备信息
     *
     * @return
     */
    public static String getDevice() {
        return Build.DEVICE;
    }

    /**
     * 获取系统版本号
     *
     * @return
     */
    public static String getDisplayVersion() {
        return Build.DISPLAY;
    }

    /**
     * 获取语言
     *
     * @return
     */
    public static String getLanguage() {
        return Locale.getDefault().getLanguage();
    }

    /**
     * 获取国家
     *
     * @param ctx
     * @return
     */
    public static String getCountry(Context ctx) {
        TelephonyManager tm = (TelephonyManager) ctx.getSystemService(Context.TELEPHONY_SERVICE);
        Locale locale = Locale.getDefault();
        return tm.getSimState() == TelephonyManager.SIM_STATE_READY ? tm.getSimCountryIso().toLowerCase(Locale.getDefault()) : locale.getCountry().toLowerCase(locale);
    }

    /**
     * 获取设备UA(WebView 的UserAgent）如果是有派设备 UserAgent后添加 +";webview_android;youpie";
     *
     * @param context
     * @return
     */
    public static String getDeviceUA(Context context) {
        String ua = "";
        try {
            //api 19 之前
            if(Build.VERSION.SDK_INT < 19){
                WebView mWebView = new WebView(context);
                ua = mWebView.getSettings().getUserAgentString();

                //销毁VebView
                mWebView.destroy();
                //WebView置为null
                mWebView = null;
            }else {
                ua = WebSettings.getDefaultUserAgent(context);
            }

            if (DevicesUtils.isYouPie() || DevicesUtils.isARM()) {
                ua += ";webview_android;youpie";
            }
        } catch (NullPointerException e) {
            return null;
        }
        return ua;
    }

    /**
     * 获取当前设备类型
     */
    public static String getDeviceType() {
        String type = getYPSN();
        if (type.startsWith(UIConsts.DeviceType.YP_DEVICE)) {
            return UIConsts.DeviceType.YP_DEVICE;
        } else if (type.startsWith(UIConsts.DeviceType.ARM_DEVICE)) {
            return UIConsts.DeviceType.ARM_DEVICE;
        } else {
            return UIConsts.DeviceType.OTHER_DEVICE;
        }
    }

    /**
     * 获取设备SN  读取的本地文件
     */
    public static String getYPSN() {
        String youpieSn = LoginUtils.getyoupieSn();
        if (org.apache.commons.lang3.StringUtils.isEmpty(youpieSn)) {
            LoginUtils.setYouPieSn(getSerial());
            return getSerial();
        } else {
            return youpieSn;
        }
    }

    /**
     * 获取有派sn 对象
     */
    public static String getyoupieSn() {
        String youpiesn = "";
        DeviceDataDTO data = new DeviceDataDTO();
        if (!fileIsExists(YOUPIE_PATH)) {
            // TODO: 2017-05-03
            return "";
        } else {//存在
            String userStr = read(YOUPIE_PATH);
            if (org.apache.commons.lang3.StringUtils.isNotEmpty(userStr)) {
                DeviceData user = JsonUtils.fromJSON(userStr, DeviceData.class);
                youpiesn = user.getDevicesSn();
            }
            return youpiesn;
        }
    }

    /**
     * 读取文件中的信息
     */
    public static String read(String filePath) {
        // 如果手机插入了SD卡，而且应用程序具有访问SD的权限
        String sbStr = null;
        try {
            if (Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
                // 获取SD卡的目录
                File sdDirFile = new File(filePath);
                //获取指定文件对应的输入流
                FileInputStream fis = new FileInputStream(sdDirFile);
                //将指定输入流包装成BufferedReader
                BufferedReader br = new BufferedReader(new InputStreamReader(fis));

                StringBuilder sb = new StringBuilder("");
                String line = null;
                while ((line = br.readLine()) != null) {
                    sb.append(line);
                }
                sbStr = sb.toString();
                return sbStr;
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return sbStr;
    }


    /**
     * 判断  存在返回fakse
     */
    public static boolean fileIsExists(String filePath) {
        try {
            File f = new File(filePath);
            if (!f.exists()) {
                return false;
            }
        } catch (Exception e) {
            // TODO: handle exception
            return false;
        }
        return true;
    }


    @SuppressLint("MissingPermission")
    private static String getSerial() {
        String serial = "NO Search";
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            return serial;
        } else {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                serial = Build.getSerial();
                return serial;
            } else {
                serial = Build.SERIAL;
                return serial;
            }
        }
    }


    /**
     * 获取设备型号
     * <p>如 MI2SC</p>
     *
     * @return 设备型号
     */
    public static String getDeviceModel() {
        String model = Build.MODEL;
        if (model != null) {
            model = model.trim();
        } else {
            model = "";
        }
        return model;
    }

    /**
     * 获取设备 MAC 地址
     * <p>需添加权限 {@code <uses-permission android:name="android.permission.ACCESS_WIFI_STATE" />}</p>
     * <p>需添加权限 {@code <uses-permission android:name="android.permission.INTERNET" />}</p>
     *
     * @return MAC 地址
     */
    @RequiresPermission(anyOf = {ACCESS_WIFI_STATE, INTERNET})
    public static String getMacAddress() {
        String macAddress = getMacAddressByWifiInfo();
        if (!"02:00:00:00:00:00".equals(macAddress)) {
            return macAddress;
        }
        macAddress = getMacAddressByNetworkInterface();
        if (!"02:00:00:00:00:00".equals(macAddress)) {
            return macAddress;
        }
        macAddress = getMacAddressByFile();
        if (!"02:00:00:00:00:00".equals(macAddress)) {
            return macAddress;
        }
        return "please open wifi";
    }

    /**
     * 获取设备 MAC 地址
     * <p>需添加权限 {@code <uses-permission android:name="android.permission.ACCESS_WIFI_STATE" />}</p>
     *
     * @return MAC 地址
     */
    @SuppressLint({"HardwareIds", "MissingPermission"})
    @RequiresPermission(ACCESS_WIFI_STATE)
    private static String getMacAddressByWifiInfo() {
        try {
            @SuppressLint("WifiManagerLeak")
            WifiManager wifi = (WifiManager) DYUtils.getContext().getSystemService(Context.WIFI_SERVICE);
            if (wifi != null) {
                WifiInfo info = wifi.getConnectionInfo();
                if (info != null) {
                    return info.getMacAddress();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "02:00:00:00:00:00";
    }

    /**
     * 获取设备 MAC 地址
     * <p>需添加权限 {@code <uses-permission android:name="android.permission.INTERNET" />}</p>
     *
     * @return MAC 地址
     */
    @RequiresPermission(INTERNET)
    private static String getMacAddressByNetworkInterface() {
        try {
            List<NetworkInterface> nis = Collections.list(NetworkInterface.getNetworkInterfaces());
            for (NetworkInterface ni : nis) {
                if (!ni.getName().equalsIgnoreCase("wlan0")) {
                    continue;
                }
                byte[] macBytes = ni.getHardwareAddress();
                if (macBytes != null && macBytes.length > 0) {
                    StringBuilder res1 = new StringBuilder();
                    for (byte b : macBytes) {
                        res1.append(String.format("%02x:", b));
                    }
                    return res1.deleteCharAt(res1.length() - 1).toString();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "02:00:00:00:00:00";
    }

    /**
     * 获取设备 MAC 地址
     *
     * @return MAC 地址
     */
    private static String getMacAddressByFile() {
        ShellUtils.CommandResult result = ShellUtils.execCommand("getprop wifi.interface", false);
        if (result.result == 0) {
            String name = result.successMsg;
            if (name != null) {
                result = ShellUtils.execCommand("cat /sys/class/net/" + name + "/address", false);
                if (result.result == 0) {
                    if (result.successMsg != null) {
                        return result.successMsg;
                    }
                }
            }
        }
        return "02:00:00:00:00:00";
    }

    /**
     * 判断设备是否 root
     *
     * @return the boolean{@code true}: 是<br>{@code false}: 否
     */
    public static boolean isDeviceRooted() {
        String su = "su";
        String[] locations = {"/system/bin/", "/system/xbin/", "/sbin/", "/system/sd/xbin/",
                "/system/bin/failsafe/", "/data/local/xbin/", "/data/local/bin/", "/data/local/"};
        for (String location : locations) {
            if (new File(location + su).exists()) {
                return true;
            }
        }
        return false;
    }

    /**
     * 关机
     * <p>需要 root 权限或者系统权限 {@code <android:sharedUserId="android.uid.system" />}</p>
     */
    public static void shutdown() {
        ShellUtils.execCommand("reboot -p", true);
        Intent intent = new Intent("android.intent.action.ACTION_REQUEST_SHUTDOWN");
        intent.putExtra("android.intent.extra.KEY_CONFIRM", false);
        DYUtils.getContext().startActivity(intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK));
    }

    /**
     * 重启
     * <p>需要 root 权限或者系统权限 {@code <android:sharedUserId="android.uid.system" />}</p>
     */
    public static void reboot() {
        ShellUtils.execCommand("reboot", true);
        Intent intent = new Intent(Intent.ACTION_REBOOT);
        intent.putExtra("nowait", 1);
        intent.putExtra("interval", 1);
        intent.putExtra("window", 0);
        DYUtils.getContext().sendBroadcast(intent);
    }

    /**
     * 重启
     * <p>需系统权限 {@code <android:sharedUserId="android.uid.system" />}</p>
     *
     * @param reason 传递给内核来请求特殊的引导模式，如"recovery"
     */
    public static void reboot(final String reason) {
        PowerManager mPowerManager =
                (PowerManager) DYUtils.getContext().getSystemService(Context.POWER_SERVICE);
        try {
            if (mPowerManager == null) {
                return;
            }
            mPowerManager.reboot(reason);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 重启到 recovery
     * <p>Requires root permission.</p>
     */
    public static void reboot2Recovery() {
        ShellUtils.execCommand("reboot recovery", true);
    }

    /**
     * 重启到 bootloader
     * <p>Requires root permission.</p>
     */
    public static void reboot2Bootloader() {
        ShellUtils.execCommand("reboot bootloader", true);
    }

}

